<?php

namespace App\Http\Controllers;

use App\DataTables\AccountDataTable;
use App\Http\Requests;
use App\Http\Requests\CreateAccountRequest;
use App\Http\Requests\UpdateAccountRequest;
use App\Repositories\AccountRepository;
use Flash;
use App\Http\Controllers\AppBaseController;
use Response;
use App\Models\GroupTree;
use Illuminate\Http\Request;
use Log;


class AccountController extends AppBaseController
{
    /** @var  AccountRepository */
    private $accountRepository;

    public function __construct(AccountRepository $accountRepo)
    {
        $this->accountRepository = $accountRepo;
    }

    /**
     * Display a listing of the Account.
     *
     * @param AccountDataTable $accountDataTable
     * @return Response
     */
    public function index(AccountDataTable $accountDataTable)
    {
        return $accountDataTable->render('accounts.index');
    }

    
    /**
     * Show the form for creating a new Account.
     *
     * @return Response
     */
    public function create()
    {
        $categories = (new \App\Models\AccountGroup())->getChildNameArray();


        //dd($categories);
        
        return view('accounts.create')->with('groupList', $categories);
    }

    private function updateBalance(& $input){
         if($input['balance_type'] == 'Cr'){
             $input['balance_cr'] = (float)$input['balance'];
             $input['balance_dr'] = 0;
         }else{
            $input['balance_cr'] = 0;
            $input['balance_dr'] = (float)$input['balance'];
         }   
    }

    /**
     * Store a newly created Account in storage.
     *
     * @param CreateAccountRequest $request
     *
     * @return Response
     */
    public function store(CreateAccountRequest $request)
    {
        $input = $request->all();

        $this->updateBalance($input);

        $account = $this->accountRepository->create($input);

        if($request->expectsJson()){
            return  $this->sendResponse( new \App\Http\Resources\Account\AccountResource($account), 'Account created successfully.');
        }else{
            Flash::success('Account saved successfully.');
            return redirect(route('accounts.index'));
        }

        
    }

    /**
     * Display the specified Account.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function show($id, Request $request)
    {
        $account = $this->accountRepository->find($id);

        $result = $this->checkEmpty($account, $request);
        if($result){
            return $result;
        }

        return view('accounts.show')->with('account', $account);
    }

    /**
     * Show the form for editing the specified Account.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $account = $this->accountRepository->find($id);

        $account->balance = ($account->op_balance==0)?$account->op_balance_dc:$account->op_balance;
        $account->balance_type = ($account->op_balance==0)?'Dr':'Cr';

        $categories = (new \App\Models\AccountGroup())->getChildNameArray();

        if (empty($account)) {
            Flash::error('Account not found');

            return redirect(route('accounts.index'));
        }

        return view('accounts.edit')->with('account', $account)->with('groupList', $categories);
    }

    /**
     * Update the specified Account in storage.
     *
     * @param  int              $id
     * @param UpdateAccountRequest $request
     *
     * @return Response
     */
    public function update($id, UpdateAccountRequest $request)
    {
        $account = $this->accountRepository->find($id);

        $result = $this->checkEmpty($account, $request);
        if($result){
            return $result;
        }

        $input = $request->all();

        Log::info('account data before balance update '. json_encode($input));

        $this->updateBalance($input);

        
        Log::info('account data after balance update '. json_encode($input));
        $account = $this->accountRepository->update($input, $id);

        if($request->expectsJson()){
            return  $this->sendResponse( new \App\Http\Resources\Account\AccountResource($account), 'Account updated successfully.');
        }else{
            Flash::success('Account updated successfully.');
            return redirect(route('accounts.index'));
        }
    }

    /**
     * Remove the specified Account from storage.
     *
     * @param  int $id
     *
     * @return Response
     */
    public function destroy($id, Request $request)
    {
        $account = $this->accountRepository->find($id);

        $result = $this->checkEmpty($account, $request);
        if($result){
            return $result;
        }

        $this->accountRepository->delete($id);

        if($request->expectsJson()){
            return  $this->sendResponseMessage('Account deleted successfully.');;
        }else{
            Flash::success('Account deleted successfully.');
        return redirect(route('accounts.index'));
        }

        
    }

    private function  checkEmpty($bondMaterialRecipe , $request){
        if (empty($bondMaterialRecipe)){
            if($request->expectsJson()){
                return $this->sendValidationError('Account not found');
            }else{
                Flash::error('Account not found');
                return redirect(route('accounts.index'));
            }
        }
        return null;
    }
}
