<?php

namespace App\Models;

use Eloquent as Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Translatable\HasTranslations;
use Log;

/**
 * Class BondMaterialRecipe
 * @package App\Models
 * @version October 11, 2019, 10:04 pm UTC
 *
 */
class BondMaterialRecipe extends SensitiveCRUDAwareModel
{
    //use SoftDeletes;
    use HasTranslations;

    public $table = 'bond_material_recipes';

    protected $primaryKey = 'material_id';

    public $incrementing = false;
    

    protected $dates = ['deleted_at'];

    //for hidden info
    protected $redacted = ['powder_cost'];
   


    public $fillable = [
        'material_id',
        'bond_id',
        'weight',
        'powder_cost',
        'packing_settings_id',
        'finishing_settings_id',
        'total_piece_packing',
        'priority',
        'bond_product_id',
        'bond_process_id'
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        "weight" => "float",
        "powder_cost" => "float"
    ];

    /**
     * Validation rules
     *
     * @var array
     */
    public static $rules = [
        'bond_name' => 'required',
        'weight' => 'required|numeric',
        'powder_cost' => 'required|numeric',
        'packing_settings' => 'required',
        'finishing_settings' => 'required',
        'total_piece_packing' => 'nullable|numeric',
        'bond_process' => 'required',
        'priority' => 'required|in:NORMAL,MEDIUM,HIGH,URGENT'
    ];

    public function packingSettings(){
        return $this->belongsTo('App\Models\ConsumableSettings','packing_settings_id');
    }
    public function finishingSettings(){
        return $this->belongsTo('App\Models\ConsumableSettings','finishing_settings_id');
    }

    public function bondproduct(){
        return $this->belongsTo('App\Models\BondProduct','bond_product_id');
    }

    public function bondProcessParam()
    {
        return $this->belongsTo('App\Models\BondProcessParams','bond_process_id');
    }

    public static function filterEle($input, $create){
        $input['packing_settings_id'] = (int)$input['packing_settings'];
        $input['finishing_settings_id'] = (int)$input['finishing_settings'];
        $input['bond_process_id'] = (int)$input['bond_process'];
        $input['bond_product_id'] = (int)$input['bond_name'];
        if($create){
            $input['material_id'] = self::generateMaterialId(5);
        }
        return $input;
    }

    public function status()
    {
        return $this->hasMany('App\Models\BondRecipeProductionStatus','material_id');
    }

    public function expectedPcs(){
        $weight = $this->bondproduct->design->piece_net_weight;
        Log::info('weight '.$weight. '  piece weight '.$this->weight);
        if($weight==0){
            return 0;
        }
        return (int) ($this->weight/$weight);
    }

    public function expectedCompletionTime($pcs){
        $bondProcess = $this->bondProcessParam;
        $expectedMould = $bondProcess->expected_mould;
        $mouldSetting = $bondProcess->findSetting('MOULD');
        if($mouldSetting->qty==0 || $expectedMould==0){
            return '0 Hour';
        }
        $time = ($pcs/$mouldSetting->qty)/$expectedMould;
        $time = number_format($time, 2);
        return $time.' Hour';
    }

    public function getPiecePerMould(){
        $bondProcess = $this->bondProcessParam;
        $mouldSetting = $bondProcess->findSetting('MOULD');
        if(empty($mouldSetting)){
            return 0;
        }
        return $mouldSetting->qty;
    }

    public static function getTotalProductionPcs($materialId){
        $material = BondMaterialRecipe::find($materialId);
        $mouldSetting = $material->bondProcessParam->findSetting('MOULD');
        $mould = BondRecipeProduction::getTotalMould($materialId);
        $piece = BondRecipeProduction::getTotalRemainingPiece($materialId);
        Log::info('materialId getTotalProductionPcs '.$materialId. ' total Mould '.$mould.' totalRemaining Piece '.$piece);
        return ($mouldSetting->qty*$mould)+$piece;
    }
}
